document.addEventListener('DOMContentLoaded', () => {
    const captureBtn = document.getElementById('capture-btn');
    const cropBtn = document.getElementById('crop-btn');
    const statusDiv = document.getElementById('status');
    const settingsView = document.getElementById('settings-view');
    const mainView = document.getElementById('main-view');
    const toggleSettingsBtn = document.getElementById('toggle-settings');
    const backBtn = document.getElementById('back-btn');
    const saveSettingsBtn = document.getElementById('save-settings');
    const apiTokenInput = document.getElementById('api-token');
    const baseUrlInput = document.getElementById('base-url');
  
    const authView = document.getElementById('auth-view');
    const loginBtn = document.getElementById('login-btn');
    const toggleSettingsAuthBtn = document.getElementById('toggle-settings-auth');
    const logoutBtn = document.getElementById('logout-btn');
    const clearHistoryBtn = document.getElementById('clear-history');

    const guestModeBtn = document.getElementById('guest-mode-btn');

    // Kayıtlı ayarları yükle
    chrome.storage.sync.get(['apiToken', 'baseUrl', 'isGuest'], (items) => {
      if (items.apiToken) apiTokenInput.value = items.apiToken;
      if (items.baseUrl) baseUrlInput.value = items.baseUrl;
      
      if (!items.apiToken && !items.isGuest) {
        showAuth(); // Token yoksa ve misafir değilse Giriş ekranını göster
      } else {
        showMain(); // Token varsa veya misafir modundaysa Ana ekranı göster
      }
    });

    // Yeni butonlar için listener'lar
    guestModeBtn.addEventListener('click', () => {
        // Misafir modunu aktif et
        chrome.storage.sync.set({ isGuest: true }, () => {
            showMain();
        });
    });

    loginBtn.addEventListener('click', () => {
        // Bağlantı sayfasını yeni sekmede aç
        chrome.tabs.create({ url: 'https://imgupla.com/connect-extension' });
    });

    toggleSettingsAuthBtn.addEventListener('click', showSettings);
  
    toggleSettingsBtn.addEventListener('click', showSettings);
    backBtn.addEventListener('click', showMain);
  
    logoutBtn.addEventListener('click', () => {
        if(confirm('Oturumu kapatmak istediğinize emin misiniz?')) {
            chrome.storage.sync.remove(['apiToken', 'isGuest'], () => {
                apiTokenInput.value = '';
                showAuth();
            });
        }
    });

    clearHistoryBtn.addEventListener('click', () => {
        if(confirm('Geçmişi temizlemek istediğinize emin misiniz?')) {
            chrome.storage.local.remove(['uploadHistory'], () => {
                loadHistory();
            });
        }
    });

    saveSettingsBtn.addEventListener('click', () => {
      const token = apiTokenInput.value.trim();
      let url = baseUrlInput.value.trim();
      
      // URL sonundaki slash'i temizle
      if (url.endsWith('/')) {
        url = url.slice(0, -1);
      }
  
      if (!token || !url) {
        statusDiv.textContent = "Tüm alanları doldurunuz!";
        statusDiv.style.color = "red";
        return;
      }
  
      chrome.storage.sync.set({ apiToken: token, baseUrl: url }, () => {
        statusDiv.textContent = "Ayarlar kaydedildi!";
        statusDiv.style.color = "green";
        setTimeout(() => {
            statusDiv.textContent = "";
            showMain();
        }, 1000);
      });
    });
  
    captureBtn.addEventListener('click', () => {
      statusDiv.textContent = "Ekran görüntüsü alınıyor...";
      statusDiv.style.color = "#666";
      captureBtn.disabled = true;
      cropBtn.disabled = true;
  
      chrome.tabs.captureVisibleTab(null, { format: 'png' }, (dataUrl) => {
        if (chrome.runtime.lastError) {
          showError("Hata: " + chrome.runtime.lastError.message);
          return;
        }
        
        uploadImage(dataUrl);
      });
    });
  
    // Crop (Alan Seçimi) Butonu
    cropBtn.addEventListener('click', () => {
        // Popup kapanacağı için mesaj gönderip işi background'a devrediyoruz
        chrome.runtime.sendMessage({ action: 'start_crop' });
        window.close(); // Popup'ı kapat
    });

    function uploadImage(dataUrl) {
      statusDiv.textContent = "Yükleniyor...";
      
      chrome.storage.sync.get(['apiToken', 'baseUrl'], (items) => {
        // Eğer token yoksa ama URL varsa devam et (misafir modu)
        // Eğer URL yoksa varsayılanı kullan
        const baseUrl = items.baseUrl || 'https://imgupla.com';
        
        // Data URL'i Blob'a çevir
        const blob = dataURItoBlob(dataUrl);
        const formData = new FormData();
        formData.append('file', blob, 'screenshot.png');
  
        const headers = {
            'Accept': 'application/json'
        };

        // Token varsa ekle
        if (items.apiToken) {
            headers['token'] = items.apiToken;
        }

        fetch(`${baseUrl}/api/upload`, {
          method: 'POST',
          headers: headers,
          body: formData
        })
        .then(response => response.json())
        .then(data => {
          if (data.result === true) {
            const link = data.url;
            const thumb = data.thumb || null; // API'den gelen thumb bilgisi
            
            navigator.clipboard.writeText(link).then(() => {
              statusDiv.innerHTML = `<span style="color:green">Yüklendi! Link kopyalandı.</span><br><a href="${link}" target="_blank" style="font-size:11px;">${link}</a>`;
            });
            saveToHistory(link, thumb); // Thumb'ı kaydet
            loadHistory();
          } else {
            showError("Yükleme başarısız: " + (data.data || "Bilinmeyen hata"));
          }
        })
        .catch(error => {
          showError("Bağlantı hatası: " + error.message);
        })
        .finally(() => {
          captureBtn.disabled = false;
        });
      });
    }
  
    function showSettings() {
      mainView.classList.add('hidden');
      authView.classList.add('hidden');
      settingsView.classList.remove('hidden');
    }
  
    function showMain() {
      settingsView.classList.add('hidden');
      authView.classList.add('hidden');
      mainView.classList.remove('hidden');
      loadHistory();
    }

    function showAuth() {
        settingsView.classList.add('hidden');
        mainView.classList.add('hidden');
        authView.classList.remove('hidden');
    }
  
    function showError(msg) {
      statusDiv.textContent = msg;
      statusDiv.style.color = "red";
      captureBtn.disabled = false;
    }
  
    // Yardımcı Fonksiyon: DataURI -> Blob
    function dataURItoBlob(dataURI) {
      const byteString = atob(dataURI.split(',')[1]);
      const mimeString = dataURI.split(',')[0].split(':')[1].split(';')[0];
      const ab = new ArrayBuffer(byteString.length);
      const ia = new Uint8Array(ab);
      for (let i = 0; i < byteString.length; i++) {
        ia[i] = byteString.charCodeAt(i);
      }
      return new Blob([ab], {type: mimeString});
    }

    function saveToHistory(url, thumbUrl = null) {
        chrome.storage.local.get(['uploadHistory'], (result) => {
            const history = result.uploadHistory || [];
            history.unshift({ 
                url: url, 
                thumb: thumbUrl || 'icons/icon48.png',
                date: new Date().toISOString() 
            });
            // Son 20 kaydı tut (arttırıldı)
            if (history.length > 20) history.pop();
            chrome.storage.local.set({ uploadHistory: history });
        });
    }

    function loadHistory() {
        chrome.storage.local.get(['uploadHistory'], (result) => {
            const history = result.uploadHistory || [];
            const historyList = document.getElementById('history-list');
            
            if (history.length === 0) {
                historyList.innerHTML = '<div class="empty-state">Henüz yükleme yapılmadı.<br>Ekran görüntüsü alarak veya sağ tık ile başlayın!</div>';
                return;
            }

            let html = '';
            history.forEach(item => {
                // Tarihi formatla
                const date = new Date(item.date);
                const dateStr = date.toLocaleDateString('tr-TR') + ' ' + date.toLocaleTimeString('tr-TR', {hour: '2-digit', minute:'2-digit'});
                
                // Thumb kontrolü (eski kayıtlarda olmayabilir)
                const thumb = item.thumb || 'icons/icon48.png';

                html += `
                    <div class="history-item">
                        <img src="${thumb}" class="history-thumb" onerror="this.src='icons/icon48.png'">
                        <div class="history-info">
                            <a href="${item.url}" target="_blank" class="history-link" title="${item.url}">${item.url}</a>
                            <div class="history-date">${dateStr}</div>
                        </div>
                        <div class="history-actions">
                            <button class="icon-btn copy-btn" data-url="${item.url}" title="Kopyala">
                                📋
                            </button>
                        </div>
                    </div>
                `;
            });
            historyList.innerHTML = html;

            // Kopyalama butonları için event listener
            document.querySelectorAll('.copy-btn').forEach(btn => {
                btn.addEventListener('click', (e) => {
                    const url = e.target.closest('.copy-btn').getAttribute('data-url');
                    navigator.clipboard.writeText(url).then(() => {
                        // İkonu değiştir
                        const originalContent = btn.innerHTML;
                        btn.innerHTML = '✅';
                        setTimeout(() => {
                            btn.innerHTML = originalContent;
                        }, 1000);
                    });
                });
            });
        });
    }
  });
