// background.js

// Eklenti yüklendiğinde Context Menu oluştur
chrome.runtime.onInstalled.addListener(() => {
  chrome.contextMenus.create({
    id: "uploadToImgupla",
    title: "Imgupla'ya Yükle",
    contexts: ["image"]
  });
});

// Context Menu tıklamalarını dinle
chrome.contextMenus.onClicked.addListener((info, tab) => {
  if (info.menuItemId === "uploadToImgupla" && info.srcUrl) {
    uploadImageUrl(info.srcUrl);
  }
});

// Popup'tan veya Content Script'ten gelen mesajları dinle
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === 'start_crop') {
      // Aktif sekmeye cropper.js'i enjekte et
      chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
        if (tabs.length === 0) return;
        chrome.scripting.executeScript({
          target: { tabId: tabs[0].id },
          files: ['cropper.js']
        });
      });
    } 
    else if (request.action === 'area_selected') {
      // Alan seçildi, ekran görüntüsü al ve kırp
      captureAndCrop(request.area);
    }
    else if (request.action === 'save_token_from_web') {
        // Web sitesinden gelen token'ı kaydet
        if (request.token) {
            chrome.storage.sync.set({ 
                apiToken: request.token,
                baseUrl: 'https://imgupla.com' // Varsayılan URL'i de ayarla
            }, () => {
                sendResponse({ success: true });
            });
            return true; // Asenkron yanıt için
        }
    }
  });
  
  async function captureAndCrop(area) {
    try {
      // 1. Tüm görünür alanı yakala
      const dataUrl = await chrome.tabs.captureVisibleTab(null, { format: 'png' });
      
      // 2. Görüntüyü kırp (OffscreenCanvas kullanımı)
      const blob = await cropImage(dataUrl, area);
  
      // 3. Yükle
      uploadImage(blob);
  
    } catch (err) {
      console.error(err);
      showNotification('Hata', 'Görüntü işlenirken hata oluştu: ' + err.message);
    }
  }
  
  async function cropImage(dataUrl, area) {
    // Veriyi Bitmap'e çevir
    const response = await fetch(dataUrl);
    const blob = await response.blob();
    const bitmap = await createImageBitmap(blob);
  
    // DPI çarpanını hesaba kat (Retina ekranlar için)
    const scale = area.devicePixelRatio || 1;
    
    // Kırpılacak alanın gerçek boyutları
    const x = area.x * scale;
    const y = area.y * scale;
    const width = area.width * scale;
    const height = area.height * scale;
  
    // OffscreenCanvas oluştur
    const canvas = new OffscreenCanvas(width, height);
    const ctx = canvas.getContext('2d');
  
    // Belirtilen alanı çiz
    ctx.drawImage(bitmap, x, y, width, height, 0, 0, width, height);
  
    // Blob olarak döndür
    return await canvas.convertToBlob({ type: 'image/png' });
  }
  
  function uploadImage(fileBlob) {
    // Ayarları al
    chrome.storage.sync.get(['apiToken', 'baseUrl'], (items) => {
      // Misafir modu kontrolü (Token yoksa da URL varsa veya default kullanılacaksa)
      const baseUrl = items.baseUrl || 'https://imgupla.com';
  
      const formData = new FormData();
      formData.append('file', fileBlob, 'cropped-screenshot.png');

      const headers = {
        'Accept': 'application/json'
      };

      if (items.apiToken) {
          headers['token'] = items.apiToken;
      }
  
      fetch(`${baseUrl}/api/upload`, {
        method: 'POST',
        headers: headers,
        body: formData
      })
      .then(response => response.json())
      .then(data => {
        if (data.result === true) {
          // Başarılı! Linki kopyala (Background'da clipboard erişimi sınırlı olabilir, ama deneyelim)
          // Not: Service worker'da navigator.clipboard genellikle çalışmaz. 
          // Bu yüzden sadece bildirim gösterip, linki bildirimde vereceğiz.
          showNotification('Yüklendi!', 'Resim başarıyla yüklendi: ' + data.url, data.url);
          saveToHistory(data.url, data.thumb); // Thumb bilgisini ekle
        } else {
          showNotification('Başarısız', 'Yükleme hatası: ' + (data.data || 'Bilinmeyen hata'));
        }
      })
      .catch(error => {
        showNotification('Hata', 'Bağlantı hatası: ' + error.message);
      });
    });
  }

  function uploadImageUrl(url) {
    chrome.storage.sync.get(['apiToken', 'baseUrl'], (items) => {
      // Misafir modu kontrolü
      const baseUrl = items.baseUrl || 'https://imgupla.com';

      const formData = new FormData();
      formData.append('file-link', url);

      const headers = {
        'Accept': 'application/json'
      };

      if (items.apiToken) {
          headers['token'] = items.apiToken;
      }

      fetch(`${baseUrl}/api/upload-link`, {
        method: 'POST',
        headers: headers,
        body: formData
      })
      .then(response => response.json())
      .then(data => {
        if (data.result === true) {
          showNotification('Yüklendi!', 'Resim başarıyla yüklendi: ' + data.url, data.url);
          saveToHistory(data.url, data.thumb); // Thumb bilgisini ekle
        } else {
          showNotification('Başarısız', 'Yükleme hatası: ' + (data.data || 'Bilinmeyen hata'));
        }
      })
      .catch(error => {
        showNotification('Hata', 'Bağlantı hatası: ' + error.message);
      });
    });
  }

  function saveToHistory(url, thumbUrl = null) {
    chrome.storage.local.get(['uploadHistory'], (result) => {
        const history = result.uploadHistory || [];
        history.unshift({ 
            url: url, 
            thumb: thumbUrl || 'icons/icon48.png', // Varsayılan ikon
            date: new Date().toISOString() 
        });
        // Son 20 kaydı tut
        if (history.length > 20) history.pop();
        chrome.storage.local.set({ uploadHistory: history });
    });
  }
  
  function showNotification(title, message, url = null) {
    chrome.notifications.create({
      type: 'basic',
      iconUrl: 'icons/icon48.png',
      title: title,
      message: message,
      priority: 2
    });
  
    // Eğer URL varsa, bildirime tıklandığında aç
    if (url) {
        // Geçici olarak URL'i saklayalım (basit bir çözüm)
        chrome.storage.local.set({ lastUrl: url });
    }
  }

// Bildirime tıklanırsa linki yeni sekmede aç
chrome.notifications.onClicked.addListener(() => {
    chrome.storage.local.get(['lastUrl'], (result) => {
        if (result.lastUrl) {
            chrome.tabs.create({ url: result.lastUrl });
        }
    });
});
